(function (global) {
    "use strict";

    // A collection of all supported modules.
    var modules = {};

    // The default return type for unknown modules.
    var defaultModule = function () { return function () { }; };

    /**
    * Require a node module.
    * @param {String} module The name/path of the module to require.
    */
    var require = function (module) {
        var fn = modules[module] || defaultModule;
        return fn.call();
    };

    // The Node.js EventEmitter class.
    var EventEmitter = function () {
        /**
         * Add an event listener.
         * @param {String} event The name of the event, such as "data", "error" etc.
         * @param {Function} listener The function to execute when the event fires.
         */
        this.addListener = function (event, listener) { return this; };

        /**
         * Adds a listener to the end of the listeners array for the specified event.
         * @param {String} event The name of the event, such as "data", "error" etc.
         * @param {Function} listener The function to execute when the event fires.
         */
        this.on = function () { return this; };

        /**
         * Adds a one time listener for the event. This listener is invoked only the next time the event is fired, after which it is removed.
         * @param {String} event The name of the event, such as "data", "error" etc.
         * @param {Function} listener The function to execute when the event fires.
         */
        this.once = function () { return this; };

        /**
         * Remove a listener from the listener array for the specified event. Caution: changes array indices in the listener array behind the listener.
         * @param {String} event The name of the event, such as "data", "error" etc.
         * @param {Function} listener The function to execute when the event fires.
         */
        this.removeListener = function () { return this; };

        /**
         * Removes all listeners, or those of the specified event. It's not a good idea to remove listeners that were added elsewhere in the code, especially when it's on an emitter that you didn't create (e.g. sockets or file streams). 
         * @param {String} [event] The name of the event, such as "data", "error" etc.
         */
        this.removeAllListeners = function () { return this; };

        /**
         * Returns an array of listeners for the specified event.
         * @param {String} event The name of the event, such as "data", "error" etc.
         */
        this.listeners = function () { return [this]; };

        /**
         * Returns an array of listeners for the specified event.
         * @param {String} event The name of the event, such as "data", "error" etc.
         */
        this.emit = function () { return true; };

        /**
         * By default EventEmitters will print a warning if more than 10 listeners are added for a particular event. This is a useful default which helps finding memory leaks. Obviously not all Emitters should be limited to 10. This function allows that to be increased.
         * @param {Number} max Set to zero for unlimited.
         */
        this.setMaxListeners = function () { };
    };

    // The Node.js Stream interface.
    var Stream = function () {
        var readable = {

            // This method will cause a stream in flowing-mode to stop emitting data events. Any data that becomes available will remain in the internal buffer.
            pause: function () { },

            /**
             * This method pulls all the data out of a readable stream, and writes it to the supplied destination, automatically managing the flow so that the destination is not overwhelmed by a fast readable stream.
             * @param {Stream} stream The destination for writing data.
             * @param {Object} [options]
             */
            pipe: function (stream) {

                if (!stream || !stream._$append) {
                    return this;
                }

                for (var prop in stream) {
                    if (prop.indexOf("_$") !== 0) {
                        this[prop] = stream[prop];
                    }
                }

                return this;
            },

            /**
            * The read() method pulls some data out of the internal buffer and returns it. If there is no data available, then it will return null.
            * @param {Number} [size] Optional argument to specify how much data to read.
            */
            read: function () { },

            // This method will cause the readable stream to resume emitting data events.
            resume: function () { },

            /**
            * Call this function to cause the stream to return strings of the specified encoding instead of Buffer objects.
            * @param {String} encoding The encoding to use.
            */
            setEncoding: function () { },

            /**
            * This method will remove the hooks set up for a previous pipe() call.
            * @param {Stream} [stream] The specific stream to unpipe.
            */
            unpipe: function () { },

            /**
            * This is useful in certain cases where a stream is being consumed by a parser, which needs to "un-consume" some data that it has optimistically pulled out of the source, so that the stream can be passed on to some other party.
            * @param {String or Buffer} chunk The specific stream to unpipe.
            */
            unshift: function () { },

            wrap: function () { }
        };

        var writable = {
            write: function (data, encoding, callback) { },
            end: function (chunk, encoding, callback) { }
        };

        var duplex = merge(readable, writable);

        EventEmitter.call(readable);
        EventEmitter.call(writable);

        return {
            Readable: function () { return readable; },
            Writable: function () { return writable; },
            Duplex: function () { return duplex; },
            Transform: function () { return duplex; }
        };
    };

    modules["events"] = function () {
        return {
            EventEmitter: new EventEmitter()
        };
    };

    modules["gulp"] = function () {
        var dependencies = {
            vfs: modules["vinyl-fs"]()
        };

        var gulp = {
            /**
             * Registers a Gulp task.
             * @param {String} name The name of the task. Tasks that you want to run from the command line should not have spaces in them.
             * @param {Array} [deps] An array of tasks to be executed and completed before your task will run.
             * @param {Function} [fn] The function that performs the task's operations. Generally this takes the form of gulp.src().pipe(someplugin()).
             */
            task: function () {
                return {
                    isRunning: false,
                    start: function () { },
                    stop: function () { },
                    doneCallback: function () { }
                };
            },

            tasks: {},

            /**
             * Checks if a task has been registered with Gulp.
             * @param {String} name The name of the task.
             */
            hasTask: function () { return true; },

            isRunning: true,
            domain: null,
            seq: [],
            doneCallback: undefined,

            dest: dependencies.vfs.dest,
            src: dependencies.vfs.src,
            watch: dependencies.vfs.watch
        };

        return gulp;
    };

    modules["gulp-coffee"] = function () {

        global.coffeeScriptConfig = function () {
            return {
                bare: true,
                shiftLine: true,
                header: "",
                sandbox: {}
            };
        };

        /**
         * Compiles CoffeeScript files to JavaScript.
         * @param {coffeeScriptConfig} [options] A configuration object for CoffeeScript.
         */
        return function () { };
    };

    modules["gulp-concat"] = function () {

        global.concatConfig = function () {
            return {
                path: "",
                newline: true,
                stat: {
                    mode: 0
                }
            };
        };

        /**
         * This will concat files by your operating systems newLine. It will take the base directory from the first file that passes through it.
         * @param {String} [filename]
         * @param {concatConfig} [options] A configuration object for concat.
         */
        return function () { };
    };

    modules["gulp-jshint"] = function () {

        global.jshintConfig = function () {
            return {
                maxerr: 500,

                bitwise: true,
                camelcase: false,
                curly: false,
                eqeqeq: true,
                forin: true,
                immed: false,
                indent: 4,

                latedef: false,
                newcap: false,
                noarg: true,
                noempty: true,
                nonew: false,
                plusplus: false,
                quotmark: false,

                undef: true,
                unused: true,
                strict: false,
                maxparams: false,
                maxdepth: false,
                maxstatements: false,
                maxcomplexity: false,
                maxlen: false,

                asi: false,
                boss: false,
                debug: true,
                eqnull: false,
                es5: false,
                esnext: false,
                moz: false,

                evil: false,
                expr: true,
                funcscope: false,
                globalstrict: false,
                iterator: false,
                lastsemic: false,
                laxbreak: true,
                laxcomma: true,
                loopfunc: false,
                multistr: false,
                proto: false,
                scripturl: false,
                shadow: false,
                sub: false,
                supernew: false,
                validthis: false,

                browser: true,
                couch: false,
                devel: true,
                dojo: false,
                jquery: true,
                mootools: false,
                node: false,
                nonstandard: false,
                prototypejs: false,
                rhino: false,
                worker: false,
                wsh: false,
                yui: false,

                globals: {}
            };
        };

        /**
         * JavaScript linting task.
         * @param {jshintConfig} [options] A configuration object for JSHint.
         */
        var jshint = function () { };

        jshint.reporter = function () { };

        /**
         * Tells JSHint to extract JavaScript from HTML files before linting (see JSHint CLI flags). Keep in mind that it doesn't override the file's content after extraction.
         * @param {String} flags The flag can be "auto", "always" or "never". The default is "auto".
         */
        jshint.extract = function () { };

        return jshint;
    };

    modules["gulp-less"] = function () {

        global.lessConfig = function () {
            return {
                /// <field name="paths" type="String, Array or Function">Specifies directories to scan for @import directives when parsing. Default value is the directory of the source, which is probably what you want.</field>
                paths: "",

                /// <field name="rootpath" type="Boolean">A path to add on to the start of every URL resource.</field>
                rootpath: "",

                /// <field name="compress" type="Boolean">Compress output by removing some whitespaces.</field>
                compress: false,

                /// <field name="yuicompress" type="Boolean" />
                yuicompress: false,

                /// <field name="plugins" type="Array">Allows passing plugins.</field>
                plugins: [],

                /// <field name="ieCompat" type="Boolean">Enforce the CSS output is compatible with Internet Explorer 8.</field>
                ieCompat: true,

                /// <field name="optimization" type="Integer">Set the parser's optimization level. The lower the number, the less nodes it will create in the tree. This could matter for debugging, or if you want to access the individual nodes in the tree.</field>
                optimization: false,

                /// <field name="strictImports" type="Boolean">Force evaluation of imports.</field>
                strictImports: false,

                /// <field name="strictMath" type="Boolean">When enabled, math is required to be in parenthesis.</field>
                strictMath: false,

                /// <field name="strictUnits" type="Boolean">When enabled, less will validate the units used (e.g. 4px/2px = 2, not 2px and 4em/2px throws an error).</field>
                strictUnits: false,

                /// <field name="syncImport" type="Boolean">Read @import'ed files synchronously from disk.</field>
                syncImport: false,

                /// <field name="dumpLineNumbers" type="Boolean">Configures -sass-debug-info support. Accepts following values: "comments", "mediaquery", "all".</field>
                dumpLineNumbers: false,

                /// <field name="relativeUrls" type="Boolean">Rewrite URLs to be relative. false: do not modify URLs.</field>
                relativeUrls: false,

                /// <field name="customFunctions" type="Object">Define custom functions to be available within your LESS stylesheets.</field>
                customFunctions: {},

                /// <field name="sourceMapBasepath" type="String">Sets the base path for the less file paths in the source map.</field>
                sourceMapBasepath: "",

                /// <field name="sourceMapRootpath" type="String">Adds this path onto the less file paths in the source map.</field>
                sourceMapRootpath: "",

                /// <field name="modifyVars" type="Object">Overrides global variables. Equivalent to  --modify-vars='VAR=VALUE'  option in less.</field>
                modifyVars: {},

                /// <field name="banner" type="String">A banner text to inject at the top of the compiled CSS file.</field>
                banner: ""
            };
        };

        /**
         * Compiles LESS files into CSS.
         * @param {lessConfig} [options] A configuration object for LESS.
         */
        return function () { };
    };

    modules["gulp-sourcemaps"] = function () {

        global.sourcemapsInitConfig = function () {
            return {
                /// <field name="loadMaps" type="Boolean">Set to true to load existing maps for source files.</field>
                loadMaps: true,
                /// <field name="debug" type="Boolean">Set this to  true  to output debug messages (e.g. about missing source content).</field>
                debug: false
            };
        };

        global.sourcemapsWriteConfig = function () {
            return {
                /// <field name="addComment" type="Boolean">By default a comment containing / referencing the source map is added. Set this to  false  to disable the comment (e.g. if you want to load the source maps by header).</field>
                addComment: true,
                /// <field name=" includeContent" type="Boolean">By default the source maps include the source code. Pass  false  to use the original files.</field>
                includeContent: false,
                /// <field name="sourceRoot" type="String">Set the path where the source files are hosted.</field>
                sourceRoot: "",
                /// <field name="sourceMappingURLPrefix" type="String">Specify a prefix to be prepended onto the source map URL when writing external source maps. Relative paths will have their leading dots stripped.</field>
                sourceMappingURLPrefix: false,
                /// <field name="debug" type="Boolean">Set this to  true  to output debug messages (e.g. about missing source content).</field>
                debug: false
            };
        };

        return {
            /**
             * Initializes the source mapping.
             * @param {sourcemapsInitConfig} [options] A configuration object for sourcemaps.
             */
            init: function () { },

            /**
             * Writes the sourcemap.
             * @param {sourcemapsWriteConfig} [options] A configuration object for sourcemaps.
             * @param {string} [path] To write external source map files, pass a path relative to the destination to  sourcemaps.write() 
             */
            write: function () { }
        };
    };

    modules["gulp-tslint"] = function () {

        global.tslintConfig = function () {
            return {
                configuration: {
                    rules: {
                        "ban": "",
                        "class-name": true,
                        "comment-format": "",
                        "curly": true,
                        "eofline": false,
                        "forin": true,
                        "indent": true,
                        "interface-name": true,
                        "jsdoc-format": true,
                        "label-position": true,
                        "label-undefined": true,
                        "max-line-length": true,
                        "no-arg": true,
                        "no-bitwise": true,
                        "no-console": true,
                        "no-construct": true,
                        "no-debugger": true,
                        "no-duplicate-key": true,
                        "no-duplicate-variable": true,
                        "no-empty": true,
                        "no-eval": true,
                        "no-string-literal": true,
                        "no-trailing-comma": true,
                        "no-trailing-whitespace": true,
                        "no-unused-expression": true,
                        "no-unused-variable": true,
                        "no-unreachable": true,
                        "no-use-before-declare": true,
                        "one-line": true,
                        "quotemark": true,
                        "radix": true,
                        "semicolon": true,
                        "triple-equals": true,
                        "typedef": true,
                        "typedef-whitespace": true,
                        "use-strict": true,
                        "variable-name": false,
                        "whitespace": true
                    }
                },
                rulesDirectory: null,
                emitError: true
            };
        };

        /**
         * Runs static code analysis on TypeScript files.
         * @param {tslintConfig} [options] A configuration object for TSLint.
         */
        var fn = function () { };

        /**
         * Specify a TSLint reporter.
         * @param {String or Function} reporter
         * @param {Object} [options] A configuration object for TSLint reporting.
         */
        fn.report = function () { };

        return fn;
    };

    modules["gulp-typescript"] = function () {

        global.typescriptConfig = function () {
            return {
                /// <field name="removeComments" type="Boolean" />
                removeComments: true,
                /// <field name="noImplicitAny" type="Boolean" />
                noImplicitAny: true,
                /// <field name="noLib" type="Boolean" />
                noLib: true,
                /// <field name="target" type="String" />
                target: "",
                /// <field name="module" type="String" />
                module: "",
                /// <field name="sourceRoot" type="String" />
                sourceRoot: "",
                /// <field name="declarationFiles" type="Boolean" />
                declarationFiles: true,
                /// <field name="noExternalResolve" type="Boolean" />
                noExternalResolve: true,
                /// <field name="sortOutput" type="Boolean" />
                sortOutput: true
            };
        };

        /**
         * Creates a TypeScript project.
         * @param {typescriptConfig} [options] A configuration object for TypeScript.
         * @param {object} [filterSettings] A filter object for TypeScript.
         * @param {function} [reporter] Specify a reporter for TypeScript.
         */
        var fn = function () {
            return {
                _$append: true,
                dts: new Stream().Readable(),
                js: new Stream().Readable()
            };
        };

        /**
         * Creates a TypeScript project.
         * @param {typescriptConfig} [options] A configuration object for TypeScript.
         */
        fn.createProject = function () { };

        /**
         * Adds a filter to the TypeScript compiler.
         * @param {Object} [options] A filter object for TypeScript.
         */
        fn.filter = function () {

        };

        fn.reporter = {
            voidReporter: function () { },
            defaultReporter: function () { },
            fullReporter: function () { }
        };

        return fn;
    };

    modules["gulp-uglify"] = function () {

        global.uglifyConfig = function () {
            return {
                /// <field name="mangle" type="Boolean" />
                mangle: true,
                /// <field name="output" type="Boolean or Object" />
                output: {},
                /// <field name="compress" type="Boolean or Object" />
                compress: true,
                /// <field name=" preserveComments" type="String" />
                preserveComments: "",
                /// <field name=" outSourceMap" type="Boolean" />
                outSourceMap: ""
            };
        };

        /**
         * Uglifies JavaScript files
         * @param {uglifyConfig} [options] A configuration object for uglify.
         */
        return function () { };
    };

    modules["glob-watcher"] = function () {
        /**
         * Watch files and do something when a file changes. This always returns an EventEmitter that emits change events.
         * @param {String} glob A single glob or array of globs that indicate which files to watch for changes.
         * @param {Object} [options] Optional configuration object, that are passed to "gaze".
         * @param {Array} tasks Names of task(s) to run when a file changes, added with "gulp.task()".
         */
        function fn() {
            return {
                end: function () { },
                add: function () { },
                remove: function () { },

                /**
                 * Hook into events.
                 * @param {String} event The name of the event, such as "data", "error" etc.
                 * @param {Function} callback The callback function to execute when the event fires.
                 */
                on: function () { }
            };
        }

        return fn;
    };

    modules["vinyl-fs"] = function () {
        var dependencies = {
            watcher: modules["glob-watcher"]()
        };

        return {
            /**
            * Emits files matching provided glob or an array of globs. Returns a stream of Vinyl files that can be piped to plugins.
            * @param {String} glob Glob or array of globs to read.
            * @param {Object} [options] Options to pass to node-glob through glob-stream.
            */
            src: function () { return new Stream().Readable(); },

            /**
             * @param {String} outFolder The path (output folder) to write files to.
             * @param {Object} [options] A configuration object.
             */
            dest: function () { return new Stream().Readable(); },

            watch: dependencies.watcher
        };
    };

    modules["stream"] = function () {
        return new Stream();
    };

    // Helper function that merges two objects
    function merge(obj1, obj2) {
        for (var attrname in obj2) { obj1[attrname] = obj2[attrname]; }
        return obj1;
    }

    intellisense.addEventListener('statementcompletion', function (event) {
        event.items = event.items.filter(function (item) {
            return item.name.indexOf("Config") === -1 || (item.name.length - item.name.indexOf("Config")) !== "Config".length;
        });
    });

    // Set global properties to make them visible in Intellisense inside gulpfile.js.
    global.require = require;

})(this);
// SIG // Begin signature block
// SIG // MIIdswYJKoZIhvcNAQcCoIIdpDCCHaACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFCZJ0MJP/7H5
// SIG // FnKMpKIBmP2RFGWGoIIYZTCCBMMwggOroAMCAQICEzMA
// SIG // AADFlkBgS/Teri4AAAAAAMUwDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTE2MDkwNzE3
// SIG // NTg1MloXDTE4MDkwNzE3NTg1MlowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBALa8M+Aljqb58BAVTntXKyhJa7NwkJf8rp9XHZKb
// SIG // oIfzMbLqBp+wS+NxkU8UZ+7Dw+66tQ8Y+LJj5YFWa/2Y
// SIG // RjAPCd81zqsiHA+IWBYpF7oZRrrkqJLpwgMVJeTeIIzf
// SIG // zO/jcCT5tibWCmSwXR+Sf8vhiJO60IaYG7LaSzmX+Bh4
// SIG // AvDMo4u9PPTtbl9jAihRVPsLdmOetDuF3NjVY3rhvp56
// SIG // uaRsqvdddZyt39A3k+xVow1UFhYISjocngW+IjIZpHQP
// SIG // L1qWJsekzXPnd+Z0mXC3NBIUiFSZMxAWH8JK88RWhv6Y
// SIG // DSfQuI5v8x4MFi2cEKVOytnz9eXxpARqZcDY69cCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBQOAfV/O8l4AlhGcTsJ
// SIG // /Jp1V4Y4xDAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQAFEt7YEhMw
// SIG // qcrsNtcklAFrxpbUMbp8i1Amjn9XX31HBtuaGjlv1riB
// SIG // SbNJb/D8uk7Dq7pbMf+DU5CMl62vzLCb8HMzF9rP78eo
// SIG // Otzt6LECPOahw3afCDFCxpq8vqKj1U7ry+wWnPdORKqh
// SIG // lxnfVGO8teBXClgVidvL453E8PAlvO+qWPNJ0pvIQv9/
// SIG // DOETBjRKeyhmMceWYBl9zaQFXlAYqO13HFxxMuwkjZl5
// SIG // tlfIo3rhmSfXyDJIYgFixHyjoMRmet7OchsKTLdkBP+j
// SIG // 8tN7hfl9gxyu+sNxUIIaolDTxQu7lKPaaEnwK5niw7rK
// SIG // ZDwCeOnW9Nm6jckSz9NG3M88MIIGBzCCA++gAwIBAgIK
// SIG // YRZoNAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYK
// SIG // CZImiZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJ
// SIG // bWljcm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9v
// SIG // dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAz
// SIG // MTI1MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IB
// SIG // DwAwggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQ
// SIG // rc7dGE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm
// SIG // 2gBr6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd
// SIG // 6udOo6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOw
// SIG // htDlKEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7Po
// SIG // IZd++NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi
// SIG // +YSnEUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1U
// SIG // n68eeEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXD
// SIG // wiALAgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/
// SIG // MB0GA1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzAL
// SIG // BgNVHQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgG
// SIG // A1UdIwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KSh
// SIG // Y6RhMF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJ
// SIG // kiaJk/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1p
// SIG // Y3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0
// SIG // eYIQea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWg
// SIG // Q6BBhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtp
// SIG // L2NybC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5j
// SIG // cmwwVAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjho
// SIG // dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRz
// SIG // L01pY3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeK
// SIG // w1wDRDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQS
// SIG // ooxtYrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ
// SIG // 7DsB7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+F
// SIG // A+mRKiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxn
// SIG // LcVRDupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+F
// SIG // ChPfwgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4w
// SIG // yrt4ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1w
// SIG // lGysOUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8cs
// SIG // u89Ds+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyz
// SIG // s7BNZ8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RI
// SIG // zCzU2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/
// SIG // m4LBJ1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HI
// SIG // jCPJZr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCs
// SIG // lLCleKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbr
// SIG // Hk/Jmu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0b
// SIG // odj6L54/LlUWa8kTo/0wggYRMIID+aADAgECAhMzAAAA
// SIG // joeRpFcaX8o+AAAAAACOMA0GCSqGSIb3DQEBCwUAMH4x
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jv
// SIG // c29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwHhcNMTYx
// SIG // MTE3MjIwOTIxWhcNMTgwMjE3MjIwOTIxWjCBgzELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEeMBwG
// SIG // A1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA0IfUQit+
// SIG // ndnGetSiw+MVktJTnZUXyVI2+lS/qxCv6cnnzCZTw8Jz
// SIG // v23WAOUA3OlqZzQw9hYXtAGllXyLuaQs5os7efYjDHmP
// SIG // 81LfQAEcwsYDnetZz3Pp2HE5m/DOJVkt0slbCu9+1jIO
// SIG // XXQSBOyeBFOmawJn+E1Zi3fgKyHg78CkRRLPA3sDxjnD
// SIG // 1CLcVVx3Qv+csuVVZ2i6LXZqf2ZTR9VHCsw43o17lxl9
// SIG // gtAm+KWO5aHwXmQQ5PnrJ8by4AjQDfJnwNjyL/uJ2hX5
// SIG // rg8+AJcH0Qs+cNR3q3J4QZgHuBfMorFf7L3zUGej15Tw
// SIG // 0otVj1OmlZPmsmbPyTdo5GPHzwIDAQABo4IBgDCCAXww
// SIG // HwYDVR0lBBgwFgYKKwYBBAGCN0wIAQYIKwYBBQUHAwMw
// SIG // HQYDVR0OBBYEFKvI1u2yFdKqjvHM7Ww490VK0Iq7MFIG
// SIG // A1UdEQRLMEmkRzBFMQ0wCwYDVQQLEwRNT1BSMTQwMgYD
// SIG // VQQFEysyMzAwMTIrYjA1MGM2ZTctNzY0MS00NDFmLWJj
// SIG // NGEtNDM0ODFlNDE1ZDA4MB8GA1UdIwQYMBaAFEhuZOVQ
// SIG // BdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWG
// SIG // Q2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMv
// SIG // Y3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0wNy0wOC5j
// SIG // cmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVo
// SIG // dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2Nl
// SIG // cnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0wNy0wOC5j
// SIG // cnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOC
// SIG // AgEARIkCrGlT88S2u9SMYFPnymyoSWlmvqWaQZk62J3S
// SIG // VwJRavq/m5bbpiZ9CVbo3O0ldXqlR1KoHksWU/PuD5rD
// SIG // BJUpwYKEpFYx/KCKkZW1v1rOqQEfZEah5srx13R7v5II
// SIG // UV58MwJeUTub5dguXwJMCZwaQ9px7eTZ56LadCwXreUM
// SIG // tRj1VAnUvhxzzSB7pPrI29jbOq76kMWjvZVlrkYtVylY
// SIG // 1pLwbNpj8Y8zon44dl7d8zXtrJo7YoHQThl8SHywC484
// SIG // zC281TllqZXBA+KSybmr0lcKqtxSCy5WJ6PimJdXjryp
// SIG // WW4kko6C4glzgtk1g8yff9EEjoi44pqDWLDUmuYx+pRH
// SIG // jn2m4k5589jTajMWUHDxQruYCen/zJVVWwi/klKoCMTx
// SIG // 6PH/QNf5mjad/bqQhdJVPlCtRh/vJQy4njpIBGPveJii
// SIG // XQMNAtjcIKvmVrXe7xZmw9dVgh5PgnjJnlQaEGC3F6tA
// SIG // E5GusBnBmjOd7jJyzWXMT0aYLQ9RYB58+/7b6Ad5B/eh
// SIG // Mzj+CZrbj3u2Or2FhrjMvH0BMLd7HaldG73MTRf3bkcz
// SIG // 1UDfasouUbi1uc/DBNM75ePpEIzrp7repC4zaikvFErq
// SIG // HsEiODUFhe/CBAANa8HYlhRIFa9+UrC4YMRStUqCt4Uq
// SIG // AEkqJoMnWkHevdVmSbwLnHhwCbwwggd6MIIFYqADAgEC
// SIG // AgphDpDSAAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3Nv
// SIG // ZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAx
// SIG // MTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
// SIG // MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
// SIG // dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
// SIG // aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01p
// SIG // Y3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIi
// SIG // MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCr8Ppy
// SIG // EBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgGOBoESbp/
// SIG // wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAk
// SIG // bK+eSZzpaF7S35tTsgosw6/ZqSuuegmv15ZZymAaBelm
// SIG // dugyUiYSL+erCFDPs0S3XdjELgN1q2jzy23zOlyhFvRG
// SIG // uuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqAT
// SIG // d76UPe/74ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7
// SIG // hhvZPrGMXeiJT4Qa8qEvWeSQOy2uM1jFtz7+MtOzAz2x
// SIG // sq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoD
// SIG // ho33X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGS
// SIG // rhwjp6lm7GEfauEoSZ1fiOIlXdMhSz5SxLVXPyQD8NF6
// SIG // Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
// SIG // 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1J
// SIG // MKerjt/sW5+v/N2wZuLBl4F77dbtS+dJKacTKKanfWeA
// SIG // 5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGFRInE
// SIG // CUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70
// SIG // lrC8RqBsmNLg1oiMCwIDAQABo4IB7TCCAekwEAYJKwYB
// SIG // BAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQBdOCqhc3
// SIG // NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIA
// SIG // QwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/
// SIG // MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO4eqnxzHRI4k0
// SIG // MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
// SIG // b0NlckF1dDIwMTFfMjAxMV8wM18yMi5jcmwwXgYIKwYB
// SIG // BQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0Nl
// SIG // ckF1dDIwMTFfMjAxMV8wM18yMi5jcnQwgZ8GA1UdIASB
// SIG // lzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
// SIG // FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3Bz
// SIG // L2RvY3MvcHJpbWFyeWNwcy5odG0wQAYIKwYBBQUHAgIw
// SIG // NB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkAXwBz
// SIG // AHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQEL
// SIG // BQADggIBAGfyhqWY4FR5Gi7T2HRnIpsLlhHhY5KZQpZ9
// SIG // 0nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj82nbY78i
// SIG // NaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcu
// SIG // KZvmPRul1LUdd5Q54ulkyUQ9eHoj8xN9ppB0g430yyYC
// SIG // RirCihC7pKkFDJvtaPpoLpWgKj8qa1hJYx8JaW5amJbk
// SIG // g/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49
// SIG // C/IIidYfwzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhR
// SIG // RVg4MnEnGn+x9Cf43iw6IGmYslmJaG5vp7d0w0AFBqYB
// SIG // Kig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKG
// SIG // wx1jNpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG
// SIG // 0QaxdR8UvmFhtfDcxhsEvt9Bxw4o7t5lL+yX9qFcltgA
// SIG // 1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
// SIG // eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//W
// SIG // syNodeav+vyL6wuA6mk7r/ww7QRMjt/fdW1jkT3RnVZO
// SIG // T7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5IRcBC
// SIG // yZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIEujCCBLYC
// SIG // AQEwgZUwfjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
// SIG // c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
// SIG // BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UE
// SIG // AxMfTWljcm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAx
// SIG // MQITMwAAAI6HkaRXGl/KPgAAAAAAjjAJBgUrDgMCGgUA
// SIG // oIHOMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwG
// SIG // CisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMCMGCSqG
// SIG // SIb3DQEJBDEWBBTlOflDhkTPWYQdW2mjLYcMXo9uhjBu
// SIG // BgorBgEEAYI3AgEMMWAwXqBEgEIATQBpAGMAcgBvAHMA
// SIG // bwBmAHQAIABWAGkAcwB1AGEAbAAgAFMAdAB1AGQAaQBv
// SIG // ACAAVwBlAGIAIABUAG8AbwBsAHOhFoAUaHR0cDovL3d3
// SIG // dy5hc3AubmV0LyAwDQYJKoZIhvcNAQEBBQAEggEAabgh
// SIG // G8nqR/Cif7HSZU9hWhldUGj1POyTGFUhbwqaE/pI217t
// SIG // 2KJbTQJ9rtZMb3RMYyT16yELFnVWYjkcSPKhR6IoeVot
// SIG // XyQeiQvVGsa6M7I34PjJekQemshaNRBBTEmUxwPPEP+U
// SIG // 8q9Et6j41Dy/PQGx3i7TCVaB+5DCuO5A2FxcZaa4suuk
// SIG // rL/OZnz9D2uIxAQ7U84V5re9IurbhLCxOL8IhL61Bjsp
// SIG // XJDiyza0OeLDi6AmM+GvKEHxcVadP1FW++b7o1RXeD/M
// SIG // pCzwfOh/lrXMSmmeSdV0rpffsQEEgDECUs19ksvwlEze
// SIG // P6chtlljX2naKcxXPQmdYOY/TQriPaGCAigwggIkBgkq
// SIG // hkiG9w0BCQYxggIVMIICEQIBATCBjjB3MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0ECEzMAAADFlkBgS/Teri4AAAAAAMUw
// SIG // CQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG
// SIG // 9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTE3MDIyNDA2MTky
// SIG // OVowIwYJKoZIhvcNAQkEMRYEFNaxtUeRu9PK/2yvO1XZ
// SIG // aaQcRt8zMA0GCSqGSIb3DQEBBQUABIIBAHZqcG7z+V+i
// SIG // ZDO4tAr9EfyYukUWgtEhJ0amBLulcNDfuNzwa5kY/qYd
// SIG // lavHxp5BymGXIKukNXgCkBSe6bvxJx2d2OV2AP1Lzxtv
// SIG // HbbR/RH3038AB7IvMo9klgg+HLyMlv32W+rI6CFkPDZY
// SIG // BLgVO98wkO3pHBsOAJJ8UP8yVeqCaaTDph3rkjlkBkCL
// SIG // WlqoTNd0Hkrov6XRhTAvQ6G69Uip3oZZBZJ6FTNCqnC6
// SIG // aHlLSMWOLf0zt563CAZEY6zCTVHPOvqujSZnaYkotSnY
// SIG // a4VQJRHoQz1KTD4kWPLmOjkhH/k6GITJzIpQ5ZSvIwif
// SIG // 2AHJFfMM/8Esgu0lJlnMtXA=
// SIG // End signature block
